<?php
/**
 * Users API Endpoint
 * Handles CRUD operations for users with htmx responses
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../components/user-list-item.php';

header('Content-Type: text/html; charset=utf-8');

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Mock users database
$users = [
    [
        'id' => 1,
        'name' => 'John Doe',
        'email' => 'john.doe@email.com',
        'role' => ROLE_STUDENT,
        'status' => 'active',
        'joined' => '2024-01-15',
        'last_active' => '2024-01-30',
        'meta' => ['Applications' => 3, 'Spent' => '$45,000']
    ],
    [
        'id' => 2,
        'name' => 'Sarah Johnson',
        'email' => 'sarah.johnson@globaledpathways.com',
        'role' => ROLE_COUNSELOR,
        'status' => 'active',
        'joined' => '2023-06-01',
        'last_active' => '2024-01-30',
        'meta' => ['Students' => 45]
    ],
    [
        'id' => 3,
        'name' => 'Michael Chen',
        'email' => 'michael.chen@globaledpathways.com',
        'role' => ROLE_ADMIN,
        'status' => 'active',
        'joined' => '2023-01-01',
        'last_active' => '2024-01-30',
        'meta' => []
    ]
];

switch ($method) {
    case 'GET':
        // Filter users
        $search = $_GET['search'] ?? '';
        $role_filter = $_GET['role'] ?? '';

        $filtered = array_filter($users, function($user) use ($search, $role_filter) {
            $match_search = empty($search) ||
                stripos($user['name'], $search) !== false ||
                stripos($user['email'], $search) !== false;

            $match_role = empty($role_filter) || $user['role'] === $role_filter;

            return $match_search && $match_role;
        });

        // Edit action - return modal content
        if ($action === 'edit' && isset($_GET['id'])) {
            $user_id = (int)$_GET['id'];
            $user = array_filter($users, fn($u) => $u['id'] === $user_id);
            $user = reset($user);

            if ($user) {
                ?>
                <div id="edit-user-modal" uk-modal="bg-close: false">
                    <div class="uk-modal-dialog">
                        <button class="uk-modal-close-default" type="button" uk-close></button>
                        <div class="uk-modal-header">
                            <h2 class="uk-modal-title">Edit User</h2>
                        </div>
                        <div class="uk-modal-body">
                            <form id="edit-user-form"
                                  hx-put="<?= BASE_URL ?>/api/users.php?id=<?= $user['id'] ?>"
                                  hx-target="#user-<?= $user['id'] ?>"
                                  hx-swap="outerHTML"
                                  _="on htmx:afterRequest UIkit.modal('#edit-user-modal').hide()">
                                <div class="uk-margin">
                                    <label class="uk-form-label">Name</label>
                                    <input class="uk-input" type="text" name="name" value="<?= e($user['name']) ?>" required>
                                </div>
                                <div class="uk-margin">
                                    <label class="uk-form-label">Email</label>
                                    <input class="uk-input" type="email" name="email" value="<?= e($user['email']) ?>" required>
                                </div>
                                <div class="uk-margin">
                                    <label class="uk-form-label">Role</label>
                                    <select class="uk-select" name="role" required>
                                        <?php foreach ($GLOBALS['role_names'] as $value => $label): ?>
                                            <option value="<?= $value ?>" <?= $user['role'] === $value ? 'selected' : '' ?>>
                                                <?= e($label) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </form>
                        </div>
                        <div class="uk-modal-footer uk-text-right">
                            <button class="uk-button uk-button-default uk-modal-close" type="button">Cancel</button>
                            <button class="uk-button uk-button-primary" type="submit" form="edit-user-form">Save Changes</button>
                        </div>
                    </div>
                </div>
                <script>UIkit.modal('#edit-user-modal').show();</script>
                <?php
            }
            exit;
        }

        // Return filtered user list
        foreach ($filtered as $user) {
            render_user_list_item($user);
        }

        if (empty($filtered)) {
            echo '<div class="uk-text-center uk-text-muted uk-padding">No users found matching your criteria.</div>';
        }
        break;

    case 'POST':
        // Create new user
        $first_name = $_POST['first_name'] ?? '';
        $last_name = $_POST['last_name'] ?? '';
        $email = $_POST['email'] ?? '';
        $role = $_POST['role'] ?? ROLE_STUDENT;

        // In production, validate and save to database
        $new_user = [
            'id' => count($users) + 1,
            'name' => trim("$first_name $last_name"),
            'email' => $email,
            'role' => $role,
            'status' => 'active',
            'joined' => date('Y-m-d'),
            'last_active' => date('Y-m-d'),
            'meta' => []
        ];

        // Return success notification
        header('HX-Trigger: ' . json_encode([
            'showMessage' => [
                'message' => 'User created successfully!',
                'status' => 'success'
            ]
        ]));

        // Return updated user list (would reload from database)
        foreach ($users as $user) {
            render_user_list_item($user);
        }
        render_user_list_item($new_user);
        break;

    case 'PUT':
        // Update user
        $user_id = (int)($_GET['id'] ?? 0);
        $name = $_POST['name'] ?? '';
        $email = $_POST['email'] ?? '';
        $role = $_POST['role'] ?? '';

        // Find and update user (mock)
        foreach ($users as &$user) {
            if ($user['id'] === $user_id) {
                $user['name'] = $name;
                $user['email'] = $email;
                $user['role'] = $role;

                header('HX-Trigger: ' . json_encode([
                    'showMessage' => [
                        'message' => 'User updated successfully!',
                        'status' => 'success'
                    ]
                ]));

                render_user_list_item($user);
                exit;
            }
        }
        break;

    case 'DELETE':
        // Delete user
        $user_id = (int)($_GET['id'] ?? 0);

        // In production, delete from database
        header('HX-Trigger: ' . json_encode([
            'showMessage' => [
                'message' => 'User deleted successfully!',
                'status' => 'warning'
            ]
        ]));

        // Return empty to remove element
        echo '';
        break;
}
