<?php
/**
 * Modal Component
 * Reusable modal dialogs
 *
 * Usage:
 * render_modal([
 *     'id' => 'add-user-modal',
 *     'title' => 'Add New User',
 *     'content' => '<form>...</form>',
 *     'footer' => '<button>Save</button>'
 * ]);
 */

function render_modal($config) {
    $id = $config['id'] ?? 'modal';
    $title = $config['title'] ?? 'Modal';
    $content = $config['content'] ?? '';
    $footer = $config['footer'] ?? '';
    $size = $config['size'] ?? ''; // 'large', 'full', or empty for default
    ?>
    <div id="<?= $id ?>" uk-modal>
        <div class="uk-modal-dialog <?= $size ? "uk-modal-{$size}" : '' ?>">
            <button class="uk-modal-close-default" type="button" uk-close></button>
            <div class="uk-modal-header">
                <h2 class="uk-modal-title"><?= e($title) ?></h2>
            </div>
            <div class="uk-modal-body">
                <?= $content ?>
            </div>
            <?php if ($footer): ?>
                <div class="uk-modal-footer uk-text-right">
                    <?= $footer ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
    <?php
}

/**
 * Render form modal for adding/editing entities
 */
function render_form_modal($id, $title, $fields, $action, $method = 'POST') {
    ob_start();
    ?>
    <form hx-<?= strtolower($method) ?>="<?= $action ?>"
          hx-target="#main-content"
          hx-swap="innerHTML"
          _="on htmx:afterRequest if event.detail.successful UIkit.modal('#<?= $id ?>').hide()">
        <?php foreach ($fields as $field): ?>
            <div class="uk-margin">
                <label class="uk-form-label"><?= e($field['label']) ?></label>
                <?php if ($field['type'] === 'select'): ?>
                    <select class="uk-select" name="<?= $field['name'] ?>" <?= $field['required'] ? 'required' : '' ?>>
                        <option value=""><?= $field['placeholder'] ?? 'Select...' ?></option>
                        <?php foreach ($field['options'] as $value => $label): ?>
                            <option value="<?= $value ?>"><?= e($label) ?></option>
                        <?php endforeach; ?>
                    </select>
                <?php elseif ($field['type'] === 'textarea'): ?>
                    <textarea class="uk-textarea" name="<?= $field['name'] ?>"
                              rows="<?= $field['rows'] ?? 3 ?>"
                              placeholder="<?= $field['placeholder'] ?? '' ?>"
                              <?= $field['required'] ? 'required' : '' ?>></textarea>
                <?php else: ?>
                    <input class="uk-input" type="<?= $field['type'] ?>"
                           name="<?= $field['name'] ?>"
                           placeholder="<?= $field['placeholder'] ?? '' ?>"
                           <?= $field['required'] ? 'required' : '' ?>>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
    </form>
    <?php
    $content = ob_get_clean();

    ob_start();
    ?>
    <button class="uk-button uk-button-default uk-modal-close" type="button">Cancel</button>
    <button class="uk-button uk-button-primary" type="submit" form="<?= $id ?>-form">
        <span uk-icon="icon: check"></span> Save
    </button>
    <?php
    $footer = ob_get_clean();

    // Re-render with form ID
    ob_start();
    ?>
    <form id="<?= $id ?>-form"
          hx-<?= strtolower($method) ?>="<?= $action ?>"
          hx-target="#main-content"
          hx-swap="innerHTML"
          _="on htmx:afterRequest if event.detail.successful UIkit.modal('#<?= $id ?>').hide()">
        <?php foreach ($fields as $field): ?>
            <div class="uk-margin">
                <label class="uk-form-label"><?= e($field['label']) ?></label>
                <?php if ($field['type'] === 'select'): ?>
                    <select class="uk-select" name="<?= $field['name'] ?>" <?= !empty($field['required']) ? 'required' : '' ?>>
                        <option value=""><?= $field['placeholder'] ?? 'Select...' ?></option>
                        <?php foreach ($field['options'] as $value => $label): ?>
                            <option value="<?= $value ?>"><?= e($label) ?></option>
                        <?php endforeach; ?>
                    </select>
                <?php elseif ($field['type'] === 'textarea'): ?>
                    <textarea class="uk-textarea" name="<?= $field['name'] ?>"
                              rows="<?= $field['rows'] ?? 3 ?>"
                              placeholder="<?= $field['placeholder'] ?? '' ?>"
                              <?= !empty($field['required']) ? 'required' : '' ?>></textarea>
                <?php else: ?>
                    <input class="uk-input" type="<?= $field['type'] ?>"
                           name="<?= $field['name'] ?>"
                           placeholder="<?= $field['placeholder'] ?? '' ?>"
                           <?= !empty($field['required']) ? 'required' : '' ?>>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
    </form>
    <?php
    $content = ob_get_clean();

    render_modal([
        'id' => $id,
        'title' => $title,
        'content' => $content,
        'footer' => $footer
    ]);
}
