<?php
/**
 * AdoxEdu Admin Portal Helper Functions
 * Reusable functions following DRY principles
 */

/**
 * Generate user initials from name
 */
function get_initials($name) {
    $words = explode(' ', $name);
    $initials = '';
    foreach ($words as $word) {
        $initials .= strtoupper(substr($word, 0, 1));
    }
    return substr($initials, 0, 2);
}

/**
 * Format currency
 */
function format_currency($amount, $symbol = '$') {
    if ($amount >= 1000000) {
        return $symbol . number_format($amount / 1000000, 1) . 'M';
    } elseif ($amount >= 1000) {
        return $symbol . number_format($amount / 1000, 0) . 'K';
    }
    return $symbol . number_format($amount, 0);
}

/**
 * Format number with K/M suffix
 */
function format_number($number) {
    if ($number >= 1000000) {
        return number_format($number / 1000000, 1) . 'M';
    } elseif ($number >= 1000) {
        return number_format($number / 1000, 1) . 'K';
    }
    return number_format($number);
}

/**
 * Format percentage
 */
function format_percentage($value, $decimals = 1) {
    return number_format($value, $decimals) . '%';
}

/**
 * Get trend indicator HTML
 */
function get_trend_indicator($value, $suffix = 'from last month') {
    $is_positive = $value >= 0;
    $class = $is_positive ? 'uk-text-success' : 'uk-text-danger';
    $icon = $is_positive ? 'arrow-up' : 'arrow-down';
    $prefix = $is_positive ? '+' : '';

    return sprintf(
        '<span class="%s"><span uk-icon="icon: %s; ratio: 0.7"></span> %s%s%% %s</span>',
        $class, $icon, $prefix, abs($value), $suffix
    );
}

/**
 * Format relative time
 */
function time_ago($datetime) {
    $now = new DateTime();
    $ago = new DateTime($datetime);
    $diff = $now->diff($ago);

    if ($diff->y > 0) return $diff->y . ' year' . ($diff->y > 1 ? 's' : '') . ' ago';
    if ($diff->m > 0) return $diff->m . ' month' . ($diff->m > 1 ? 's' : '') . ' ago';
    if ($diff->d > 0) return $diff->d . ' day' . ($diff->d > 1 ? 's' : '') . ' ago';
    if ($diff->h > 0) return $diff->h . ' hour' . ($diff->h > 1 ? 's' : '') . ' ago';
    if ($diff->i > 0) return $diff->i . ' minute' . ($diff->i > 1 ? 's' : '') . ' ago';
    return 'just now';
}

/**
 * Get role badge HTML
 */
function get_role_badge($role) {
    $colors = [
        ROLE_SUPER_ADMIN => 'primary',
        ROLE_ADMIN => 'secondary',
        ROLE_SENIOR_COUNSELOR => 'warning',
        ROLE_COUNSELOR => 'success',
        ROLE_STUDENT => 'default'
    ];

    $color = $colors[$role] ?? 'default';
    $label = $GLOBALS['role_names'][$role] ?? ucfirst($role);

    return sprintf('<span class="uk-label uk-label-%s">%s</span>', $color, strtolower($label));
}

/**
 * Get status badge HTML
 */
function get_status_badge($status, $type = 'application') {
    $statuses = $type === 'payment' ? $GLOBALS['payment_statuses'] : $GLOBALS['application_statuses'];
    $config = $statuses[$status] ?? ['label' => ucfirst($status), 'color' => 'default'];

    return sprintf(
        '<span class="uk-label uk-label-%s">%s</span>',
        $config['color'],
        $config['label']
    );
}

/**
 * Check if user has permission
 */
function has_permission($permission, $user_role = null) {
    $role = $user_role ?? $GLOBALS['current_user']['role'];

    if ($role === ROLE_SUPER_ADMIN) return true;

    $permissions = $GLOBALS['role_permissions'][$role] ?? [];
    return in_array($permission, $permissions) || in_array('all_system_access', $permissions);
}

/**
 * Generate avatar HTML
 */
function get_avatar($name, $size = 40, $image = null) {
    $initials = get_initials($name);

    if ($image) {
        return sprintf(
            '<img src="%s" alt="%s" class="uk-border-circle" style="width: %dpx; height: %dpx; object-fit: cover;">',
            htmlspecialchars($image), htmlspecialchars($name), $size, $size
        );
    }

    return sprintf(
        '<div class="uk-border-circle uk-flex uk-flex-center uk-flex-middle uk-background-muted" style="width: %dpx; height: %dpx; font-size: %dpx; font-weight: 500;">%s</div>',
        $size, $size, $size * 0.4, $initials
    );
}

/**
 * Sanitize output
 */
function e($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Get icon colors for stat cards
 */
function get_stat_icon_style($color) {
    $colors = [
        'primary' => ['bg' => '#e3f2fd', 'fg' => '#1976d2'],
        'success' => ['bg' => '#e8f5e9', 'fg' => '#388e3c'],
        'warning' => ['bg' => '#fff3e0', 'fg' => '#f57c00'],
        'danger' => ['bg' => '#ffebee', 'fg' => '#d32f2f'],
        'purple' => ['bg' => '#f3e5f5', 'fg' => '#7b1fa2'],
        'cyan' => ['bg' => '#e0f7fa', 'fg' => '#0097a7']
    ];

    $c = $colors[$color] ?? $colors['primary'];
    return sprintf('background-color: %s; color: %s;', $c['bg'], $c['fg']);
}
