<?php
require_once __DIR__ . '/../includes/bootstrap.php';

$students = [
    ['id' => 1, 'name' => 'John Doe', 'email' => 'john.doe@email.com', 'applications' => 3, 'status' => 'active', 'progress' => 75, 'next_deadline' => '2024-02-05', 'urgency' => 'high'],
    ['id' => 2, 'name' => 'Sarah Chen', 'email' => 'sarah.chen@email.com', 'applications' => 2, 'status' => 'active', 'progress' => 60, 'next_deadline' => '2024-02-10', 'urgency' => 'medium'],
    ['id' => 3, 'name' => 'Michael Rodriguez', 'email' => 'michael.r@email.com', 'applications' => 4, 'status' => 'active', 'progress' => 90, 'next_deadline' => '2024-02-15', 'urgency' => 'low'],
    ['id' => 4, 'name' => 'Emma Wilson', 'email' => 'emma.w@email.com', 'applications' => 2, 'status' => 'new', 'progress' => 25, 'next_deadline' => '2024-03-01', 'urgency' => 'low'],
    ['id' => 5, 'name' => 'David Kim', 'email' => 'david.kim@email.com', 'applications' => 3, 'status' => 'active', 'progress' => 85, 'next_deadline' => '2024-02-08', 'urgency' => 'medium']
];

$status_config = [
    'active' => ['label' => 'Active', 'color' => '#32d296'],
    'new' => ['label' => 'New Student', 'color' => '#667eea'],
    'inactive' => ['label' => 'Inactive', 'color' => '#94a3b8']
];

$urgency_config = [
    'high' => ['color' => '#f0506e', 'icon' => 'fa-exclamation-circle'],
    'medium' => ['color' => '#faa05a', 'icon' => 'fa-clock'],
    'low' => ['color' => '#32d296', 'icon' => 'fa-check-circle']
];
?>

<style>
.student-card {
    background: white;
    border-radius: 12px;
    padding: 20px;
    margin-bottom: 16px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    transition: all 0.2s ease;
    border-left: 4px solid transparent;
    cursor: pointer;
}

.student-card:hover {
    box-shadow: 0 4px 16px rgba(0, 0, 0, 0.12);
    transform: translateY(-2px);
    border-left-color: #667eea;
}

.student-card.high-urgency {
    border-left-color: #f0506e;
}

.student-card.medium-urgency {
    border-left-color: #faa05a;
}

.student-card.low-urgency {
    border-left-color: #32d296;
}

.student-avatar {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    font-size: 18px;
    flex-shrink: 0;
}

.student-name {
    font-size: 16px;
    font-weight: 600;
    color: #1a1a1a;
    margin-bottom: 4px;
}

.student-email {
    font-size: 13px;
    color: #64748b;
    margin-bottom: 6px;
}

.student-meta {
    font-size: 12px;
    color: #94a3b8;
    display: flex;
    align-items: center;
    gap: 12px;
    flex-wrap: wrap;
}

.status-badge-modern {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 6px;
    font-size: 11px;
    font-weight: 600;
    color: white;
}

.progress-circle {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    background: conic-gradient(
        var(--progress-color) calc(var(--progress) * 1%),
        #e2e8f0 0
    );
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
}

.progress-circle::before {
    content: '';
    position: absolute;
    width: 48px;
    height: 48px;
    border-radius: 50%;
    background: white;
}

.progress-value {
    position: relative;
    z-index: 1;
    font-weight: 700;
    font-size: 14px;
    color: #1a1a1a;
}

.action-button {
    padding: 8px 16px;
    border-radius: 8px;
    font-size: 13px;
    font-weight: 600;
    border: none;
    cursor: pointer;
    transition: all 0.2s ease;
    display: inline-flex;
    align-items: center;
    gap: 6px;
}

.action-button-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.action-button-primary:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
}

.action-button-secondary {
    background: #f1f5f9;
    color: #475569;
}

.action-button-secondary:hover {
    background: #e2e8f0;
}

.search-container {
    position: relative;
}

.search-input {
    padding: 10px 16px 10px 40px;
    border: 2px solid #e2e8f0;
    border-radius: 8px;
    font-size: 14px;
    width: 250px;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.search-icon {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    color: #94a3b8;
}

.filter-buttons {
    display: flex;
    gap: 8px;
}

.filter-button {
    padding: 8px 16px;
    border-radius: 8px;
    font-size: 13px;
    font-weight: 600;
    border: 2px solid #e2e8f0;
    background: white;
    color: #64748b;
    cursor: pointer;
    transition: all 0.2s ease;
}

.filter-button:hover, .filter-button.active {
    border-color: #667eea;
    color: #667eea;
    background: #f0f4ff;
}

@media (max-width: 959px) {
    .student-card {
        padding: 16px;
    }

    .search-input {
        width: 100%;
    }

    .filter-buttons {
        flex-wrap: wrap;
    }

    .progress-circle {
        width: 50px;
        height: 50px;
    }

    .progress-circle::before {
        width: 40px;
        height: 40px;
    }

    .progress-value {
        font-size: 12px;
    }
}
</style>

<!-- Header with Search and Actions -->
<div class="uk-flex uk-flex-between uk-flex-middle uk-margin-bottom mobile-stack">
    <h3 class="uk-margin-remove mobile-mb">
        <i class="fa-slab fa-solid fa-users"></i> My Students (<?= count($students) ?>)
    </h3>
    <div class="uk-flex uk-flex-middle mobile-stack" style="gap: 12px;">
        <div class="search-container mobile-full">
            <i class="fa-slab fa-solid fa-search search-icon"></i>
            <input class="search-input mobile-full" type="search" placeholder="Search students...">
        </div>
        <button class="action-button action-button-primary mobile-full" uk-toggle="target: #add-student-modal">
            <i class="fa-slab fa-solid fa-plus"></i> Add Student
        </button>
    </div>
</div>

<!-- Filter Buttons -->
<div class="filter-buttons uk-margin-bottom">
    <button class="filter-button active"><i class="fa-slab fa-solid fa-users"></i> All Students</button>
    <button class="filter-button"><i class="fa-slab fa-solid fa-user-check"></i> Active</button>
    <button class="filter-button"><i class="fa-slab fa-solid fa-user-plus"></i> New</button>
    <button class="filter-button"><i class="fa-slab fa-solid fa-exclamation-circle"></i> Urgent</button>
</div>

<!-- Student Cards -->
<?php foreach ($students as $student):
    $status = $status_config[$student['status']];
    $urgency = $urgency_config[$student['urgency']];
    $initials = strtoupper(substr($student['name'], 0, 1) . substr(strstr($student['name'], ' '), 1, 1));

    // Determine progress color
    $progress_color = $student['progress'] >= 75 ? '#32d296' : ($student['progress'] >= 50 ? '#faa05a' : '#f0506e');
?>
<div class="student-card <?= $student['urgency'] ?>-urgency"
     hx-get="<?= BASE_URL ?>/pages/student-detail.php?id=<?= $student['id'] ?>"
     hx-target="#main-content"
     hx-swap="innerHTML"
     hx-push-url="<?= BASE_URL ?>/student-detail.php?id=<?= $student['id'] ?>">
    <div class="uk-flex uk-flex-between uk-flex-top">
        <!-- Student Info -->
        <div class="uk-flex uk-flex-middle min-w-0" style="flex: 1; gap: 16px;">
            <div class="student-avatar"><?= $initials ?></div>
            <div class="min-w-0" style="flex: 1;">
                <div class="student-name">
                    <?= e($student['name']) ?>
                    <span class="status-badge-modern" style="background: <?= $status['color'] ?>;"><?= $status['label'] ?></span>
                </div>
                <div class="student-email"><?= e($student['email']) ?></div>
                <div class="student-meta">
                    <span>
                        <i class="fa-slab fa-solid fa-file-alt"></i> <?= $student['applications'] ?> applications
                    </span>
                    <span style="color: <?= $urgency['color'] ?>;">
                        <i class="fa-slab fa-solid <?= $urgency['icon'] ?>"></i> Due: <?= e($student['next_deadline']) ?>
                    </span>
                </div>
            </div>
        </div>

        <!-- Progress and Actions -->
        <div class="uk-flex uk-flex-column uk-flex-middle uk-margin-left" style="flex-shrink: 0; gap: 12px;">
            <div class="progress-circle mobile-hide" style="--progress: <?= $student['progress'] ?>; --progress-color: <?= $progress_color ?>;">
                <span class="progress-value"><?= $student['progress'] ?>%</span>
            </div>
            <div class="uk-flex" style="gap: 8px;">
                <button class="action-button action-button-primary">
                    <i class="fa-slab fa-solid fa-eye"></i> View
                </button>
                <button class="action-button action-button-secondary" onclick="event.stopPropagation(); window.location.href='messages.php?student=<?= $student['id'] ?>'">
                    <i class="fa-slab fa-solid fa-envelope"></i> Message
                </button>
            </div>
        </div>
    </div>
</div>
<?php endforeach; ?>

<!-- Add Student Modal -->
<div id="add-student-modal" uk-modal>
    <div class="uk-modal-dialog uk-modal-body">
        <button class="uk-modal-close-default" type="button" uk-close></button>
        <h3 class="uk-modal-title"><i class="fa-slab fa-solid fa-user-plus"></i> Add New Student</h3>
        <form>
            <div class="uk-margin">
                <label class="uk-form-label">Full Name</label>
                <input class="uk-input" type="text" placeholder="Enter student name">
            </div>
            <div class="uk-margin">
                <label class="uk-form-label">Email</label>
                <input class="uk-input" type="email" placeholder="student@email.com">
            </div>
            <div class="uk-margin">
                <label class="uk-form-label">Status</label>
                <select class="uk-select">
                    <option>New Student</option>
                    <option>Active</option>
                    <option>Inactive</option>
                </select>
            </div>
            <div class="uk-margin uk-text-right">
                <button class="uk-button uk-button-default uk-modal-close uk-margin-small-right" type="button">Cancel</button>
                <button class="uk-button uk-button-primary" type="submit">
                    <i class="fa-slab fa-solid fa-plus"></i> Add Student
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Simple client-side filtering
document.querySelectorAll('.filter-button').forEach(button => {
    button.addEventListener('click', function() {
        document.querySelectorAll('.filter-button').forEach(b => b.classList.remove('active'));
        this.classList.add('active');
    });
});

// Simple search functionality
document.querySelector('.search-input').addEventListener('input', function(e) {
    const searchTerm = e.target.value.toLowerCase();
    document.querySelectorAll('.student-card').forEach(card => {
        const name = card.querySelector('.student-name').textContent.toLowerCase();
        const email = card.querySelector('.student-email').textContent.toLowerCase();
        if (name.includes(searchTerm) || email.includes(searchTerm)) {
            card.style.display = '';
        } else {
            card.style.display = 'none';
        }
    });
});
</script>
