<?php
require_once __DIR__ . '/../includes/bootstrap.php';

// Get filter parameters
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? 'all';
$sort = $_GET['sort'] ?? 'deadline';

$applications = [
    ['id' => 'APP-2024-001', 'university' => 'Stanford University', 'program' => 'Computer Science (MS)', 'deadline' => '2024-01-15', 'submitted' => '2024-01-10', 'status' => 'under_review', 'progress' => 60, 'counselor' => 'Dr. Sarah Johnson', 'days_until_deadline' => -7],
    ['id' => 'APP-2024-002', 'university' => 'University of Cambridge', 'program' => 'Computer Science (PhD)', 'deadline' => '2024-02-01', 'submitted' => '2024-01-20', 'status' => 'interview_scheduled', 'progress' => 80, 'counselor' => 'Prof. Michael Chen', 'days_until_deadline' => 10],
    ['id' => 'APP-2024-003', 'university' => 'University of Toronto', 'program' => 'Data Science (MS)', 'deadline' => '2024-01-30', 'submitted' => '2024-01-25', 'status' => 'accepted', 'progress' => 100, 'counselor' => 'Dr. Emily Brown', 'days_until_deadline' => -22],
    ['id' => 'APP-2024-004', 'university' => 'MIT', 'program' => 'Artificial Intelligence (MS)', 'deadline' => '2024-02-15', 'submitted' => null, 'status' => 'draft', 'progress' => 35, 'counselor' => 'Dr. Sarah Johnson', 'days_until_deadline' => 24],
    ['id' => 'APP-2024-005', 'university' => 'Harvard University', 'program' => 'Business Analytics (MBA)', 'deadline' => '2024-02-10', 'submitted' => null, 'status' => 'incomplete', 'progress' => 55, 'counselor' => 'Dr. Emily Brown', 'days_until_deadline' => 19]
];

// Filter applications
$filtered_applications = array_filter($applications, function($app) use ($search, $status_filter) {
    $matches_search = empty($search) ||
                     stripos($app['university'], $search) !== false ||
                     stripos($app['program'], $search) !== false ||
                     stripos($app['id'], $search) !== false;

    $matches_status = $status_filter === 'all' || $app['status'] === $status_filter;

    return $matches_search && $matches_status;
});

// Sort applications
usort($filtered_applications, function($a, $b) use ($sort) {
    switch ($sort) {
        case 'deadline':
            return strcmp($a['deadline'], $b['deadline']);
        case 'university':
            return strcmp($a['university'], $b['university']);
        case 'status':
            return strcmp($a['status'], $b['status']);
        case 'progress':
            return $b['progress'] - $a['progress'];
        default:
            return 0;
    }
});

// Status badge helper
function get_application_status_badge($status) {
    $badges = [
        'draft' => '<span class="uk-label uk-label-muted"><i class="fa-solid fa-pencil fa-xs"></i> Draft</span>',
        'incomplete' => '<span class="uk-label" style="background: #f0506e;"><i class="fa-solid fa-exclamation-triangle fa-xs"></i> Incomplete</span>',
        'under_review' => '<span class="uk-label" style="background: #faa05a;"><i class="fa-solid fa-clock fa-xs"></i> Under Review</span>',
        'interview_scheduled' => '<span class="uk-label uk-label-warning"><i class="fa-solid fa-calendar-check fa-xs"></i> Interview Scheduled</span>',
        'accepted' => '<span class="uk-label uk-label-success"><i class="fa-solid fa-check-circle fa-xs"></i> Accepted</span>',
        'rejected' => '<span class="uk-label uk-label-danger"><i class="fa-solid fa-times-circle fa-xs"></i> Rejected</span>',
        'waitlisted' => '<span class="uk-label" style="background: #32d296;"><i class="fa-solid fa-hourglass-half fa-xs"></i> Waitlisted</span>'
    ];
    return $badges[$status] ?? '<span class="uk-label">' . e(ucwords(str_replace('_', ' ', $status))) . '</span>';
}

// Deadline warning helper
function get_deadline_warning($days_until) {
    if ($days_until < 0) {
        return null; // Past deadline
    } elseif ($days_until <= 3) {
        return '<span class="uk-text-danger"><i class="fa-solid fa-exclamation-circle"></i> Due in ' . $days_until . ' day' . ($days_until !== 1 ? 's' : '') . '!</span>';
    } elseif ($days_until <= 7) {
        return '<span class="uk-text-warning"><i class="fa-solid fa-clock"></i> Due in ' . $days_until . ' days</span>';
    }
    return null;
}

// Get unique statuses for filter
$all_statuses = array_unique(array_column($applications, 'status'));
?>

<style>
.application-item {
    position: relative;
    z-index: 1;
    transition: all 0.3s ease;
}

.application-item:hover {
    z-index: 10;
    box-shadow: 0 5px 20px rgba(0,0,0,0.08);
}

.application-item:has(.uk-open) {
    z-index: 10;
}

.application-item .uk-dropdown {
    z-index: 1020 !important;
}

.application-progress-circle {
    position: relative;
    width: 60px;
    height: 60px;
}

.application-progress-circle svg {
    transform: rotate(-90deg);
}

.application-progress-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    font-size: 14px;
    font-weight: bold;
}

.mobile-stack {
    flex-direction: column;
    align-items: flex-start !important;
    gap: 12px;
}

.mobile-mb {
    margin-bottom: 0 !important;
}

@media (min-width: 960px) {
    .mobile-stack {
        flex-direction: row;
        align-items: center !important;
    }
}

.application-header-section {
    display: flex;
    align-items: center;
    gap: 16px;
}

.application-university-logo {
    width: 56px;
    height: 56px;
    border-radius: 8px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: bold;
    font-size: 20px;
    flex-shrink: 0;
}

.application-info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
    gap: 16px;
    margin-top: 16px;
}

@media (max-width: 959px) {
    .application-info-grid {
        grid-template-columns: 1fr 1fr;
    }
}

.filter-buttons {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.filter-buttons button {
    white-space: nowrap;
}

.sort-select {
    min-width: 150px;
}
</style>

<!-- Header with Search and Actions -->
<div class="uk-flex uk-flex-between uk-flex-middle uk-margin-bottom mobile-stack">
    <h3 class="uk-margin-remove mobile-mb">My Applications</h3>
    <button class="uk-button uk-button-primary uk-button-small" uk-toggle="target: #new-application-modal">
        <i class="fa-solid fa-plus"></i> New Application
    </button>
</div>

<!-- Search and Filters -->
<div class="uk-card uk-card-default uk-card-body uk-margin-bottom">
    <div class="uk-grid uk-grid-small" uk-grid>
        <!-- Search -->
        <div class="uk-width-expand@s">
            <div class="uk-inline uk-width-1-1">
                <span class="uk-form-icon" uk-icon="icon: search"></span>
                <input class="uk-input uk-form-small"
                       type="search"
                       name="search"
                       placeholder="Search by university, program, or ID..."
                       value="<?= e($search) ?>"
                       hx-get="<?= BASE_URL ?>/pages/applications.php"
                       hx-trigger="keyup changed delay:300ms, search"
                       hx-target="#applications-list"
                       hx-include="[name='status'], [name='sort']"
                       _="on htmx:beforeRequest
                          if event.detail.pathInfo.requestPath contains 'search='
                            set $statusFilter to 'all'
                            set #status-all.checked to true">
            </div>
        </div>

        <!-- Sort -->
        <div class="uk-width-auto@s">
            <select class="uk-select uk-form-small sort-select"
                    name="sort"
                    hx-get="<?= BASE_URL ?>/pages/applications.php"
                    hx-trigger="change"
                    hx-target="#applications-list"
                    hx-include="[name='search'], [name='status']">
                <option value="deadline" <?= $sort === 'deadline' ? 'selected' : '' ?>>Sort by Deadline</option>
                <option value="university" <?= $sort === 'university' ? 'selected' : '' ?>>Sort by University</option>
                <option value="status" <?= $sort === 'status' ? 'selected' : '' ?>>Sort by Status</option>
                <option value="progress" <?= $sort === 'progress' ? 'selected' : '' ?>>Sort by Progress</option>
            </select>
        </div>
    </div>

    <!-- Status Filter -->
    <div class="uk-margin-top">
        <div class="uk-text-small uk-text-muted uk-margin-small-bottom">Filter by Status:</div>
        <div class="filter-buttons">
            <label class="uk-button uk-button-small <?= $status_filter === 'all' ? 'uk-button-primary' : 'uk-button-default' ?>">
                <input class="uk-radio" type="radio" name="status" value="all" <?= $status_filter === 'all' ? 'checked' : '' ?> id="status-all"
                       hx-get="<?= BASE_URL ?>/pages/applications.php"
                       hx-trigger="change"
                       hx-target="#applications-list"
                       hx-include="[name='search'], [name='sort']"> All
            </label>
            <label class="uk-button uk-button-small <?= $status_filter === 'draft' ? 'uk-button-primary' : 'uk-button-default' ?>">
                <input class="uk-radio" type="radio" name="status" value="draft" <?= $status_filter === 'draft' ? 'checked' : '' ?>
                       hx-get="<?= BASE_URL ?>/pages/applications.php"
                       hx-trigger="change"
                       hx-target="#applications-list"
                       hx-include="[name='search'], [name='sort']"> <i class="fa-solid fa-pencil"></i> Draft
            </label>
            <label class="uk-button uk-button-small <?= $status_filter === 'incomplete' ? 'uk-button-primary' : 'uk-button-default' ?>">
                <input class="uk-radio" type="radio" name="status" value="incomplete" <?= $status_filter === 'incomplete' ? 'checked' : '' ?>
                       hx-get="<?= BASE_URL ?>/pages/applications.php"
                       hx-trigger="change"
                       hx-target="#applications-list"
                       hx-include="[name='search'], [name='sort']"> <i class="fa-solid fa-exclamation-triangle"></i> Incomplete
            </label>
            <label class="uk-button uk-button-small <?= $status_filter === 'under_review' ? 'uk-button-primary' : 'uk-button-default' ?>">
                <input class="uk-radio" type="radio" name="status" value="under_review" <?= $status_filter === 'under_review' ? 'checked' : '' ?>
                       hx-get="<?= BASE_URL ?>/pages/applications.php"
                       hx-trigger="change"
                       hx-target="#applications-list"
                       hx-include="[name='search'], [name='sort']"> <i class="fa-solid fa-clock"></i> Under Review
            </label>
            <label class="uk-button uk-button-small <?= $status_filter === 'interview_scheduled' ? 'uk-button-primary' : 'uk-button-default' ?>">
                <input class="uk-radio" type="radio" name="status" value="interview_scheduled" <?= $status_filter === 'interview_scheduled' ? 'checked' : '' ?>
                       hx-get="<?= BASE_URL ?>/pages/applications.php"
                       hx-trigger="change"
                       hx-target="#applications-list"
                       hx-include="[name='search'], [name='sort']"> <i class="fa-solid fa-calendar-check"></i> Interview
            </label>
            <label class="uk-button uk-button-small <?= $status_filter === 'accepted' ? 'uk-button-primary' : 'uk-button-default' ?>">
                <input class="uk-radio" type="radio" name="status" value="accepted" <?= $status_filter === 'accepted' ? 'checked' : '' ?>
                       hx-get="<?= BASE_URL ?>/pages/applications.php"
                       hx-trigger="change"
                       hx-target="#applications-list"
                       hx-include="[name='search'], [name='sort']"> <i class="fa-solid fa-check-circle"></i> Accepted
            </label>
        </div>
    </div>
</div>

<!-- Applications List -->
<div id="applications-list">
    <?php if (empty($filtered_applications)): ?>
        <div class="uk-card uk-card-default uk-card-body uk-text-center">
            <i class="fa-solid fa-inbox fa-3x uk-text-muted"></i>
            <p class="uk-text-muted uk-margin-small-top">
                <?= empty($search) ? 'No applications found.' : 'No applications match your search.' ?>
            </p>
            <?php if (!empty($search)): ?>
                <button class="uk-button uk-button-default uk-button-small"
                        hx-get="<?= BASE_URL ?>/pages/applications.php"
                        hx-target="#applications-list"
                        hx-vals='{"search": "", "status": "all"}'>
                    Clear Search
                </button>
            <?php endif; ?>
        </div>
    <?php else: ?>
        <?php foreach ($filtered_applications as $app): ?>
        <div class="application-item uk-card uk-card-default uk-card-body uk-margin-bottom">
            <!-- Header Section -->
            <div class="uk-flex uk-flex-between uk-flex-top uk-margin-bottom">
                <div class="application-header-section" style="flex: 1; min-width: 0;">
                    <!-- University Logo/Icon -->
                    <div class="application-university-logo">
                        <?= strtoupper(substr($app['university'], 0, 2)) ?>
                    </div>

                    <!-- University & Program Info -->
                    <div style="flex: 1; min-width: 0;">
                        <h4 class="uk-margin-remove uk-text-truncate"><?= e($app['university']) ?></h4>
                        <div class="uk-text-muted uk-text-small uk-text-truncate"><?= e($app['program']) ?></div>
                        <div class="uk-text-meta uk-text-small uk-margin-small-top">
                            ID: <code class="uk-text-small"><?= e($app['id']) ?></code>
                        </div>
                    </div>
                </div>

                <!-- Actions Dropdown -->
                <div style="flex-shrink: 0; margin-left: 16px;">
                    <button class="uk-button uk-button-default uk-button-small" type="button">
                        <i class="fa-solid fa-ellipsis-vertical"></i>
                    </button>
                    <div uk-dropdown="mode: click; pos: bottom-right">
                        <ul class="uk-nav uk-dropdown-nav">
                            <li class="uk-nav-header">Actions</li>
                            <li><a href="<?= BASE_URL ?>/pages/application-detail.php?id=<?= e($app['id']) ?>"
                                   hx-get="<?= BASE_URL ?>/pages/application-detail.php?id=<?= e($app['id']) ?>"
                                   hx-target="#main-content"
                                   hx-swap="innerHTML"><i class="fa-solid fa-eye fa-fw"></i> View Details</a></li>
                            <?php if ($app['status'] === 'draft' || $app['status'] === 'incomplete'): ?>
                            <li><a href="#"><i class="fa-solid fa-edit fa-fw"></i> Continue Editing</a></li>
                            <?php endif; ?>
                            <li><a href="#" uk-toggle="target: #track-status-modal"><i class="fa-solid fa-chart-line fa-fw"></i> Track Status</a></li>
                            <li><a href="#" uk-toggle="target: #upload-documents-modal"><i class="fa-solid fa-cloud-upload fa-fw"></i> Upload Documents</a></li>
                            <li><a href="#" uk-toggle="target: #contact-counselor-modal"><i class="fa-solid fa-envelope fa-fw"></i> Contact Counselor</a></li>
                            <li class="uk-nav-divider"></li>
                            <li><a href="#" uk-toggle="target: #export-pdf-modal"><i class="fa-solid fa-file-pdf fa-fw"></i> Export PDF</a></li>
                            <?php if ($app['status'] !== 'accepted' && $app['status'] !== 'rejected'): ?>
                            <li><a href="#" class="uk-text-danger" uk-toggle="target: #withdraw-application-modal"><i class="fa-solid fa-times fa-fw"></i> Withdraw Application</a></li>
                            <?php endif; ?>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Status and Progress Row -->
            <div class="uk-flex uk-flex-between uk-flex-middle uk-margin-bottom">
                <div>
                    <?= get_application_status_badge($app['status']) ?>
                </div>
                <div class="uk-text-right">
                    <!-- Progress Circle (Desktop) -->
                    <div class="uk-visible@s" style="display: inline-block;">
                        <div class="application-progress-circle">
                            <svg width="60" height="60">
                                <circle cx="30" cy="30" r="25" fill="none" stroke="#e5e5e5" stroke-width="4"/>
                                <circle cx="30" cy="30" r="25" fill="none"
                                        stroke="<?= $app['progress'] === 100 ? '#32d296' : ($app['progress'] >= 60 ? '#faa05a' : '#f0506e') ?>"
                                        stroke-width="4"
                                        stroke-dasharray="<?= $app['progress'] * 1.57 ?>, 157"
                                        stroke-linecap="round"/>
                            </svg>
                            <div class="application-progress-text"><?= $app['progress'] ?>%</div>
                        </div>
                    </div>
                    <!-- Progress Bar (Mobile) -->
                    <div class="uk-hidden@s uk-width-1-1">
                        <div class="uk-flex uk-flex-between uk-text-small uk-margin-small-bottom">
                            <span class="uk-text-muted">Progress</span>
                            <span class="uk-text-bold"><?= $app['progress'] ?>%</span>
                        </div>
                        <progress class="uk-progress" value="<?= $app['progress'] ?>" max="100"></progress>
                    </div>
                </div>
            </div>

            <!-- Info Grid -->
            <div class="application-info-grid">
                <div>
                    <div class="uk-text-meta uk-text-small">Deadline</div>
                    <div class="uk-text-bold uk-text-small"><?= date('M d, Y', strtotime($app['deadline'])) ?></div>
                    <?php if ($warning = get_deadline_warning($app['days_until_deadline'])): ?>
                        <div class="uk-text-small uk-margin-small-top"><?= $warning ?></div>
                    <?php endif; ?>
                </div>
                <div>
                    <div class="uk-text-meta uk-text-small">Submitted</div>
                    <div class="uk-text-bold uk-text-small">
                        <?= $app['submitted'] ? date('M d, Y', strtotime($app['submitted'])) : '<span class="uk-text-muted">Not submitted</span>' ?>
                    </div>
                </div>
                <div>
                    <div class="uk-text-meta uk-text-small">Counselor</div>
                    <div class="uk-text-bold uk-text-small"><?= e($app['counselor']) ?></div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="uk-margin-top uk-flex uk-flex-wrap" style="gap: 8px;">
                <a href="<?= BASE_URL ?>/pages/application-detail.php?id=<?= e($app['id']) ?>"
                   hx-get="<?= BASE_URL ?>/pages/application-detail.php?id=<?= e($app['id']) ?>"
                   hx-target="#main-content"
                   hx-swap="innerHTML"
                   class="uk-button uk-button-primary uk-button-small">
                    <i class="fa-solid fa-eye"></i> View Details
                </a>
                <?php if ($app['status'] === 'draft' || $app['status'] === 'incomplete'): ?>
                <button class="uk-button uk-button-default uk-button-small">
                    <i class="fa-solid fa-edit"></i> Continue Application
                </button>
                <?php else: ?>
                <button class="uk-button uk-button-default uk-button-small" uk-toggle="target: #track-status-modal">
                    <i class="fa-solid fa-chart-line"></i> Track Status
                </button>
                <?php endif; ?>
            </div>
        </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>

<!-- New Application Modal -->
<div id="new-application-modal" uk-modal>
    <div class="uk-modal-dialog">
        <button class="uk-modal-close-default" type="button" uk-close></button>
        <div class="uk-modal-header">
            <h2 class="uk-modal-title"><i class="fa-solid fa-plus-circle"></i> Start New Application</h2>
        </div>
        <div class="uk-modal-body">
            <form class="uk-form-stacked">
                <div class="uk-margin">
                    <label class="uk-form-label">University *</label>
                    <select class="uk-select" required>
                        <option value="">Select university...</option>
                        <option>Stanford University</option>
                        <option>MIT</option>
                        <option>Harvard University</option>
                        <option>University of Cambridge</option>
                        <option>University of Oxford</option>
                        <option>University of Toronto</option>
                    </select>
                </div>
                <div class="uk-margin">
                    <label class="uk-form-label">Program *</label>
                    <select class="uk-select" required>
                        <option value="">Select program...</option>
                        <option>Computer Science (MS)</option>
                        <option>Computer Science (PhD)</option>
                        <option>Engineering (MS)</option>
                        <option>Business Administration (MBA)</option>
                        <option>Data Science (MS)</option>
                        <option>Artificial Intelligence (MS)</option>
                    </select>
                </div>
                <div class="uk-margin">
                    <label class="uk-form-label">Target Start Term *</label>
                    <select class="uk-select" required>
                        <option value="">Select term...</option>
                        <option>Fall 2024</option>
                        <option>Spring 2025</option>
                        <option>Fall 2025</option>
                    </select>
                </div>
            </form>
        </div>
        <div class="uk-modal-footer uk-text-right">
            <button class="uk-button uk-button-default uk-modal-close" type="button">Cancel</button>
            <button class="uk-button uk-button-primary" type="button">Start Application</button>
        </div>
    </div>
</div>

<!-- Track Status Modal -->
<div id="track-status-modal" uk-modal>
    <div class="uk-modal-dialog">
        <button class="uk-modal-close-default" type="button" uk-close></button>
        <div class="uk-modal-header">
            <h2 class="uk-modal-title"><i class="fa-solid fa-chart-line"></i> Application Status</h2>
        </div>
        <div class="uk-modal-body">
            <ul class="uk-list uk-list-striped">
                <li class="uk-flex uk-flex-between">
                    <span><i class="fa-solid fa-check uk-text-success"></i> Application Submitted</span>
                    <span class="uk-text-muted">Jan 10, 2024</span>
                </li>
                <li class="uk-flex uk-flex-between">
                    <span><i class="fa-solid fa-check uk-text-success"></i> Documents Verified</span>
                    <span class="uk-text-muted">Jan 15, 2024</span>
                </li>
                <li class="uk-flex uk-flex-between">
                    <span><i class="fa-solid fa-clock uk-text-warning"></i> Under Review</span>
                    <span class="uk-text-muted">In Progress</span>
                </li>
                <li class="uk-flex uk-flex-between uk-text-muted">
                    <span><i class="fa-solid fa-hourglass"></i> Interview</span>
                    <span>Pending</span>
                </li>
                <li class="uk-flex uk-flex-between uk-text-muted">
                    <span><i class="fa-solid fa-flag-checkered"></i> Decision</span>
                    <span>Pending</span>
                </li>
            </ul>
        </div>
        <div class="uk-modal-footer uk-text-right">
            <button class="uk-button uk-button-default uk-modal-close" type="button">Close</button>
        </div>
    </div>
</div>

<!-- Upload Documents Modal -->
<div id="upload-documents-modal" uk-modal>
    <div class="uk-modal-dialog">
        <button class="uk-modal-close-default" type="button" uk-close></button>
        <div class="uk-modal-header">
            <h2 class="uk-modal-title"><i class="fa-solid fa-cloud-upload"></i> Upload Documents</h2>
        </div>
        <div class="uk-modal-body">
            <form class="uk-form-stacked">
                <div class="uk-margin">
                    <label class="uk-form-label">Document Type *</label>
                    <select class="uk-select" required>
                        <option value="">Select type...</option>
                        <option>Transcript</option>
                        <option>Statement of Purpose</option>
                        <option>Letter of Recommendation</option>
                        <option>Resume/CV</option>
                        <option>Test Scores</option>
                        <option>Other</option>
                    </select>
                </div>
                <div class="uk-margin">
                    <label class="uk-form-label">File *</label>
                    <div uk-form-custom="target: true">
                        <input type="file" required>
                        <input class="uk-input uk-form-width-large" type="text" placeholder="Select file..." disabled>
                    </div>
                    <div class="uk-text-small uk-text-muted uk-margin-small-top">Max 10MB. PDF, DOC, DOCX, JPG, PNG</div>
                </div>
            </form>
        </div>
        <div class="uk-modal-footer uk-text-right">
            <button class="uk-button uk-button-default uk-modal-close" type="button">Cancel</button>
            <button class="uk-button uk-button-primary" type="button">Upload</button>
        </div>
    </div>
</div>

<!-- Contact Counselor Modal -->
<div id="contact-counselor-modal" uk-modal>
    <div class="uk-modal-dialog">
        <button class="uk-modal-close-default" type="button" uk-close></button>
        <div class="uk-modal-header">
            <h2 class="uk-modal-title"><i class="fa-solid fa-envelope"></i> Contact Counselor</h2>
        </div>
        <div class="uk-modal-body">
            <div class="uk-margin-bottom">
                <span class="uk-text-muted">Your counselor:</span> <span class="uk-text-bold">Sarah Johnson</span>
            </div>
            <form class="uk-form-stacked">
                <div class="uk-margin">
                    <label class="uk-form-label">Subject *</label>
                    <input class="uk-input" type="text" placeholder="Message subject" required>
                </div>
                <div class="uk-margin">
                    <label class="uk-form-label">Message *</label>
                    <textarea class="uk-textarea" rows="5" placeholder="Your message..." required></textarea>
                </div>
            </form>
        </div>
        <div class="uk-modal-footer uk-text-right">
            <button class="uk-button uk-button-default uk-modal-close" type="button">Cancel</button>
            <button class="uk-button uk-button-primary" type="button">Send Message</button>
        </div>
    </div>
</div>

<!-- Export PDF Modal -->
<div id="export-pdf-modal" uk-modal>
    <div class="uk-modal-dialog">
        <button class="uk-modal-close-default" type="button" uk-close></button>
        <div class="uk-modal-header">
            <h2 class="uk-modal-title"><i class="fa-solid fa-file-pdf"></i> Export to PDF</h2>
        </div>
        <div class="uk-modal-body">
            <form class="uk-form-stacked">
                <div class="uk-margin">
                    <label class="uk-form-label">Include</label>
                    <div class="uk-margin-small">
                        <label><input class="uk-checkbox" type="checkbox" checked> Application Details</label>
                    </div>
                    <div class="uk-margin-small">
                        <label><input class="uk-checkbox" type="checkbox" checked> Status History</label>
                    </div>
                    <div class="uk-margin-small">
                        <label><input class="uk-checkbox" type="checkbox" checked> Uploaded Documents List</label>
                    </div>
                    <div class="uk-margin-small">
                        <label><input class="uk-checkbox" type="checkbox"> Counselor Notes</label>
                    </div>
                </div>
            </form>
        </div>
        <div class="uk-modal-footer uk-text-right">
            <button class="uk-button uk-button-default uk-modal-close" type="button">Cancel</button>
            <button class="uk-button uk-button-primary" type="button">Export PDF</button>
        </div>
    </div>
</div>

<!-- Withdraw Application Modal -->
<div id="withdraw-application-modal" uk-modal>
    <div class="uk-modal-dialog">
        <button class="uk-modal-close-default" type="button" uk-close></button>
        <div class="uk-modal-header">
            <h2 class="uk-modal-title"><i class="fa-solid fa-exclamation-triangle"></i> Withdraw Application</h2>
        </div>
        <div class="uk-modal-body">
            <div class="uk-alert uk-alert-danger">
                <p><strong>Warning:</strong> This action cannot be undone. Withdrawing your application will permanently remove it from consideration.</p>
            </div>
            <form class="uk-form-stacked">
                <div class="uk-margin">
                    <label class="uk-form-label">Reason for withdrawal *</label>
                    <select class="uk-select" required>
                        <option value="">Select reason...</option>
                        <option>Accepted elsewhere</option>
                        <option>Changed plans</option>
                        <option>Financial reasons</option>
                        <option>Personal reasons</option>
                        <option>Other</option>
                    </select>
                </div>
                <div class="uk-margin">
                    <label class="uk-form-label">Additional comments (Optional)</label>
                    <textarea class="uk-textarea" rows="3" placeholder="Any additional information..."></textarea>
                </div>
            </form>
        </div>
        <div class="uk-modal-footer uk-text-right">
            <button class="uk-button uk-button-default uk-modal-close" type="button">Cancel</button>
            <button class="uk-button uk-button-danger" type="button">Withdraw Application</button>
        </div>
    </div>
</div>
